﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Schedulers.DateTimeValidators;
using BMS.Utils;
using System.Threading;

namespace BMS.Schedulers
{
    public class Scheduler : IEquatable<Scheduler>
    {
        private List<CommandType> commandTypes = new List<CommandType>();
        private static readonly BmsLogger logger = new BmsLogger("Scheduler: ");
        private struct CommandType
        {
            public CommandType(ICommand command, Type type)
            {
                Command = command;
                Type = type;
            }

            public readonly ICommand Command;
            public readonly Type Type;
        }

        public Scheduler(String id)
        {
            if (string.IsNullOrEmpty(id))
                throw new ArgumentException();
            Id = id;
        }

        public string Id { get; private set; }
        public string Name { get; set; }
        public TimeZoneInfo TimeZone { get; set; }
        public IDateTimeValidator DateTimeValidator { get; set; }

        public void RemoveCommand<T>() where T : ICommand
        {
            Type type = typeof(T);
            int index = commandTypes.FindIndex(s => s.Type == type);
            if (index != -1)
                commandTypes.RemoveAt(index);
        }

        public void SetCommand<T>(T command) where T : ICommand
        {
            Type type = typeof(T);
            CommandType commandType = new CommandType(command, type);
            int index = commandTypes.FindIndex(s => s.Type == type);
            if (index == -1)
                commandTypes.Add(commandType);
            else
                commandTypes[index] = commandType;
        }

        public bool Equals(Scheduler other)
        {
            if (other == null) return false;
            return Id == other.Id;
        }

        public override int GetHashCode()
        {
            return Id.GetHashCode();
        }

        public override bool Equals(object obj)
        {
            return Equals(obj as Scheduler);
        }

        private DateTime GetDateTime(DateTime utc)
        {
            return TimeZoneInfo.ConvertTimeFromUtc(utc, TimeZone);
        }

        public void Init(DateTime utcNow)
        {
            DateTime dateTime = GetDateTime(utcNow);
            DateTimeValidator.Init(dateTime);
        }

        public bool IsValid(DateTime utcNow)
        {
            DateTime dateTime = GetDateTime(utcNow);
            return DateTimeValidator.IsValid(dateTime);
        }

        public void Execute()
        {
            foreach (CommandType commandType in commandTypes)
            {
                try
                {
                    commandType.Command.Run(this);
                }
                catch (Exception e)
                {
                    logger.LogError(e.ToString());
                }
            }
        }
    }
}
